//////////////////////////////////////////////////////////////////
//
// Header file for the Planar Motor Controller library for RTX64 and KingStar
// Version Number: 06.03.039.00
// Release Date: 25/03/2025 (DD/MM/YYYY)
// Change Log: - Added GetFlywayConnectErrorCounts
//             - Added GetPMNetErrorCounts
//             - Added GetAllAccidentLocations
//
//////////////////////////////////////////////////////////////////

#pragma once

#include "PMC_Enums.h"
#include "PMC_Structs.h"
#include <chrono>

#include "ksapi.h"

#if defined UNDER_RTSS
#if defined RTX64_EXPORTS
#define __PMC_RTX64_Cpp_API __declspec(dllexport)
#else
#define __PMC_RTX64_Cpp_API __declspec(dllimport)
#endif
#else
#if defined DLL64_EXPORTS
#define __PMC_RTX64_Cpp_API __declspec(dllexport)
#else
#define __PMC_RTX64_Cpp_API __declspec(dllimport)
#endif
#endif

namespace PMC
{
	struct PMCDBType
	{
		PMC_USINT XbotState[255];
		PMC_USINT XbotStateFieldbusRaw[200];
		PMC_USINT StereoID[255];
		PMC_USINT XbotStatus[255];
		PMC_BOOL XIDScan[255];
		PMC_REAL PosX[255];
		PMC_REAL PosY[255];
		PMC_REAL PosZ[255];
		PMC_REAL PosRx[255];
		PMC_REAL PosRy[255];
		PMC_REAL PosRz[255];
		PMC_REAL FeedbackX[256];
		PMC_REAL FeedbackY[256];
		PMC_REAL FeedbackZ[256];
		PMC_REAL FeedbackRx[256];
		PMC_REAL FeedbackRy[256];
		PMC_REAL FeedbackRz[256];
		PMC_USINT FeedbackXID[150];
		PMC_ULINT PresentTime_ns[256];
		PMC_REAL PastX[256];
		PMC_REAL PastY[256];
		PMC_REAL PastRz[256];
		PMC_ULINT PastTime_ns[256];		
	};

	// This class is exported from the PMC_RTX64_Cpp_API.dll
	class __PMC_RTX64_Cpp_API PMC_API
	{
	private:
		PMC_BYTE DataFromPMC[400];
		PMC_BYTE DataToPMC[400];
		PMC_UINT PMC_Index;
		HANDLE hTimer;
		PMC_DINT RTAPI RtTimerThreadStart(PMC_DINT cycle_time_microsec);
		PMC_DINT RTAPI RtTimerThreadStop(void);
		static void StaticTimerHandler(PVOID context);
		void TimerHandler(PVOID context);

		PMC_BOOL PMC_found;
		HANDLE fieldbus_mutex;
		HANDLE cmd_semaphore;
		PMC_BOOL cmd_active;	
		PMCDBType PMCDB;
		PMC_BYTE ConfiguredFeedbackStreamAxis;
		FEEDBACKTYPECONFIGURED ConfiguredFeedbackType;
		void ClearCmdInputArea(void);
		/// <summary>
		/// Background work for PMC communication, should be called once per Jobtask. Must only be called in JobTask
		/// </summary>
		void AutoRefresh(void);
	public:
		PMC_API(void);
		~PMC_API(void);		
		/// <summary>
		/// Finds all the PMCs on the EtherCAT network and returns their IO indexes
		/// </summary>
		/// <returns></returns>
		static FindPMCsRtn FindPMCs(void);
		/// <summary>
		/// Connects to PMC. Must be called before using PMC commands. Automatically connects to first PMC found on EtherCAT
		/// </summary>
		/// <param name="PMC_IO_Index">The IO index of the PMC to connect to</param>
		/// <param name="cycle_time_microsec">The cycle time of the EtherCAT bus in microseconds</param>
		/// <returns>Error code: 0-no error</returns>
		PMCRTN ConnectToPMC(PMC_UINT PMC_IO_Index,PMC_UDINT cycle_time_microsec);
		/// <summary>
		/// Disconnects from PMC. Should be called once finished using PMC commands
		/// </summary>
		void DisconnectFromPMC();
		
		/// <summary>
		/// Measures the 1 ms full duplex throughput of the fieldbus (takes 3-4 ms to finish)
		/// </summary>
		/// <returns></returns>
		ThroughputTestRtn ThroughputTest();
		/// <summary>
		/// Move single xbot in 6 degrees of freedom (fixed velocity)
		/// </summary>
		/// <param name="cmdLB">command label (0-65535)</param>
		/// <param name="XbotID">xbot ID (1-255)</param>
		/// <param name="PosX">Target Position X, in meters</param>
		/// <param name="PosY">Target Position Y, in meters</param>
		/// <param name="PosZ">Target Position Z, in meters</param>
		/// <param name="PosRx">Target Position Rx, in radians</param>
		/// <param name="PosRy">Target Position Ry, in radians</param>
		/// <param name="PosRz">Target Position Rz, in radians</param>
		/// <returns>PMC return value + travel time in seconds</returns>
		MotionRtn SixDMotion(PMC_UINT cmdLB, PMC_USINT XbotID, PMC_REAL PosX, PMC_REAL PosY, PMC_REAL PosZ, PMC_REAL PosRx, PMC_REAL PosRy, PMC_REAL PosRz);
		/// <summary>
		/// Move single xbot in 6 degrees of freedom (user set velocity)
		/// </summary>
		/// <param name="cmdLB">label to be associated with this command in the PMC(0-65535)</param>
		/// <param name="XbotID">ID# of the xbot to move (1-255)</param>
		/// <param name="PosX">x position in meters</param>
		/// <param name="PosY">y position in meters</param>
		/// <param name="PosZ">z position in meters</param>
		/// <param name="PosRx">rx position in radians</param>
		/// <param name="PosRy">ry position in radians</param>
		/// <param name="PosRz">rz position in radians</param>
		/// <param name="LongAxisAcc">The long axis acc in m/s^2</param>
		/// <param name="LongAxisVel">The long axis vel in m/s</param>
		/// <param name="ZVel">The z vel in m/s</param>
		/// <param name="RxVel">The rx vel in radian/s</param>
		/// <param name="RyVel">The ry vel in radian/s</param>
		/// <param name="RzVel">The rz vel in radian/s</param>
		/// <returns></returns>
		MotionRtn SixDMotionVelocity(PMC_UINT cmdLB, PMC_USINT XbotID, PMC_REAL PosX, PMC_REAL PosY, PMC_REAL PosZ, PMC_REAL PosRx, PMC_REAL PosRy, PMC_REAL PosRz, PMC_REAL LongAxisAcc, PMC_REAL LongAxisVel, PMC_REAL ZVel, PMC_REAL RxVel, PMC_REAL RyVel, PMC_REAL RzVel);	
		/// <summary>
		/// Move single xbot in 6 degrees of freedom (user set velocity)(based on previously uploaded compensation data)
		/// </summary>
		/// <param name="cmdLB">label to be associated with this command in the PMC(0-65535)</param>
		/// <param name="XbotID">ID# of the xbot to move (1-255)</param>
		/// <param name="PosX">x position in meters</param>
		/// <param name="PosY">y position in meters</param>
		/// <param name="PosZ">z position in meters</param>
		/// <param name="PosRx">rx position in radians</param>
		/// <param name="PosRy">ry position in radians</param>
		/// <param name="PosRz">rz position in radians</param>
		/// <param name="LongAxisAcc">The long axis acc in m/s^2</param>
		/// <param name="LongAxisVel">The long axis vel in m/s</param>
		/// <param name="ZVel">The z vel in m/s</param>
		/// <param name="RxVel">The rx vel in radian/s</param>
		/// <param name="RyVel">The ry vel in radian/s</param>
		/// <param name="RzVel">The rz vel in radian/s</param>
		/// <returns></returns>
		MotionRtn SixDMotionVelocity_Comp(PMC_UINT cmdLB, PMC_USINT XbotID, PMC_REAL PosX, PMC_REAL PosY, PMC_REAL PosZ, PMC_REAL PosRx, PMC_REAL PosRy, PMC_REAL PosRz, PMC_REAL LongAxisAcc, PMC_REAL LongAxisVel, PMC_REAL ZVel, PMC_REAL RxVel, PMC_REAL RyVel, PMC_REAL RzVel);
		/// <summary>
		/// Command 1 - 12 xbots to follow trajectories
		/// </summary>
		/// <param name="cmdLB">The command label (0-65535)</param>
		/// <param name="nXbots">number of xbots commanded to follow trajectories (up to 12)</param>
		/// <param name="XbotID">pointer to array of ID#s of xbots to follow trajectories</param>
		/// <param name="TrajID">pointer to array of ID#s of the trajectories that the xbots are to follow</param>
		/// <returns></returns>
		MotionRtn ActivateTrajectory(PMC_UINT cmdLB, PMC_USINT nXbots, PMC_USINT* XbotID, PMC_USINT* TrajID);
		/// <summary>
		/// Command single xbot to follow a circular arc defined by arc center and arc angle
		/// </summary>
		/// <param name="cmdLB">label to be associated with this command in the PMC</param>
		/// <param name="XbotID">ID# of the xbot to move (1-255)</param>
		/// <param name="Mode">0:absolute;1: relative</param>
		/// <param name="Dir">0:clockwise;1:counter-clockwise</param>
		/// <param name="EndVel">Velocity at the end of the motion in meters/second</param>
		/// <param name="MaxVel">Maximum velocity for this motion in meters/second</param>
		/// <param name="MaxAcc">Maximum acceleration for this motion in meters/second^2</param>
		/// <param name="CenterX">X location of the arc center in meters</param>
		/// <param name="CenterY">Y location of the arc center in meters</param>
		/// <param name="Angle">Arc angle in radians</param>
		/// <returns></returns>
		MotionRtn ArcMotionCenter(PMC_UINT cmdLB, PMC_USINT XbotID, POSITIONMODE Mode, ARCDIRECTION Dir, PMC_REAL EndVel, PMC_REAL MaxVel, PMC_REAL MaxAcc, PMC_REAL CenterX, PMC_REAL CenterY, PMC_REAL Angle);
		/// <summary>
		/// Command single xbots to follow a circular arc defined by target position and arc radius 
		/// </summary>
		/// <param name="cmdLB">label to be associated with this command in the PMC</param>
		/// <param name="XbotID">ID# of the xbot to move (1-255)</param>
		/// <param name="Mode">0:absolute;1: relative</param>
		/// <param name="AType">0: Minor Arc;1: Major Arc</param>
		/// <param name="Dir">0:clockwise;1:counter-clockwise</param>
		/// <param name="EndVel">Velocity at the end of the motion in meters/second.</param>
		/// <param name="MaxVel">Maximum velocity for this motion in meters/second.</param>
		/// <param name="MaxAcc">Maximum acceleration for this motion in meters/second^2.</param>
		/// <param name="PosX">X target position in meters</param>
		/// <param name="PosY">Y target position in meters</param>
		/// <param name="Radius">Arc radius in meters</param>
		/// <returns></returns>
		MotionRtn ArcMotionTarget(PMC_UINT cmdLB, PMC_USINT XbotID, POSITIONMODE Mode, ARCTYPE AType, ARCDIRECTION Dir, PMC_REAL EndVel, PMC_REAL MaxVel, PMC_REAL MaxAcc, PMC_REAL PosX, PMC_REAL PosY, PMC_REAL Radius);
		/// <summary>
		/// Activate all Xbots
		/// </summary>
		/// <param name="IDMode">0: full auto, 1: activate to Landed state, 2: activate to Discove mode, 3: exit Discover mode</param>
		/// <returns>
		/// PMC return indicating whether command is successful
		/// </returns>
		PMCRTN ActivateXbots(PMC_USINT IDMode);
		/// <summary>
		/// Gets all the xbots in all the stations
		/// </summary>
		/// <param name="StateFilterCount">number xbot states to filter for</param>
		/// <param name="XbotStateFilters">xbot states to filter for</param>
		/// <returns></returns>
		AppDesign_GetAllStationXbotsRtn AppDesign_GetAllStationXbots(PMC_USINT StateFilterCount, PMC_USINT* XbotStateFilters);		
		/// <summary>
		/// Gets all the xbots in any state in all the stations
		/// </summary>
		/// <returns></returns>
		AppDesign_GetAllStationXbotsRtn AppDesign_GetAnyStationXbots();		
		/// <summary>
		/// gets the ID of a xbot in any state in a station's bay
		/// </summary>
		/// <param name="StationID">Station ID (>0)</param>
		/// <param name="BayID">Bay ID (>0)</param>
		/// <returns></returns>
		AppDesign_GetBayXbotIDRtn AppDesign_GetBayAnyXbot(PMC_USINT StationID, PMC_USINT BayID);		
		/// <summary>
		/// gets the ID of an idle xbot in a station's bay
		/// </summary>
		/// <param name="StationID">Station ID (>0)</param>
		/// <param name="BayID">Bay ID (>0)</param>
		/// <returns></returns>
		AppDesign_GetBayXbotIDRtn AppDesign_GetBayIdleXbot(PMC_USINT StationID, PMC_USINT BayID);
		/// <summary>
		/// gets the ID of the xbot in a station's bay
		/// </summary>
		/// <param name="StationID">Station ID (>0)</param>
		/// <param name="BayID">Bay ID (>0)</param>
		/// <param name="StateFilterCount">number xbot states to filter for</param>
		/// <param name="XbotStateFilters">xbot states to filter for</param>
		/// <returns></returns>
		AppDesign_GetBayXbotIDRtn AppDesign_GetBayXbotID(PMC_USINT StationID, PMC_USINT BayID, PMC_USINT StateFilterCount, PMC_USINT* XbotStateFilters);		
		/// <summary>
		/// reads an application designer file (as a raw binary) from the PMC
		/// </summary>
		/// <param name="DesignID">design ID to read</param>
		/// <returns></returns>
		AppDesign_GetDesignRtn AppDesign_GetDesign(PMC_USINT DesignID);
		/// <summary>
		/// Get the IDs of all xbots in any state in a station
		/// </summary>
		/// <param name="StationID">Station ID (>0)</param>
		/// <returns></returns>
		AppDesign_GetStationXbotIDsRtn AppDesign_GetStationAnyXbot(PMC_USINT StationID);		
		/// <summary>
		/// Get the IDs of all idle xbots in a station
		/// </summary>
		/// <param name="StationID">Station ID (>0)</param>
		/// <returns></returns>
		AppDesign_GetStationXbotIDsRtn AppDesign_GetStationIdleXbot(PMC_USINT StationID);
		/// <summary>
		/// Get the IDs of all xbots in a station
		/// </summary>
		/// <param name="StationID">Station ID (>0)</param>
		/// <param name="StateFilterCount">number xbot states to filter for</param>
		/// <param name="XbotStateFilters">xbot states to filter for</param>
		/// <returns></returns>
		AppDesign_GetStationXbotIDsRtn AppDesign_GetStationXbotIDs(PMC_USINT StationID, PMC_USINT StateFilterCount, PMC_USINT* XbotStateFilters);
		/// <summary>
		/// Get an xbot's target station
		/// </summary>
		/// <param name="XbotID">ID of the xbot</param>
		/// <returns></returns>
		AppDesign_GetXbotTargetRtn AppDesign_GetXbotTarget(PMC_USINT XbotID);
		/// <summary>
		/// Sends an application designer file (as a raw binary) to the PMC
		/// </summary>
		/// <param name="nBytes">number of bytes in the application designer file (up to 500,000 bytes)</param>
		/// <param name="DesignID">design ID to save the file under</param>
		/// <param name="Bytes">the application designer file as a raw binary</param>
		/// <returns></returns>
		PMCRTN AppDesign_SaveDesign(PMC_DINT nBytes, PMC_USINT DesignID, PMC_BYTE* Bytes);
		/// <summary>
		/// send an xbot to an application designer station
		/// </summary>
		/// <param name="XbotID">the xbot to send</param>
		/// <param name="StationID">the station to send the xbot to</param>
		/// <param name="BayID">the station bay to send the xbot to</param>
		/// <param name="WaitForFreeBay">0: xbot moves to station bay immediately, 1: xbot waits for the station bay to be free before moving</param>
		/// <returns></returns>
		PMCRTN AppDesign_SendXbotToStation(PMC_USINT XbotID, PMC_USINT StationID, PMC_USINT BayID, PMC_USINT WaitForFreeBay);
		/// <summary>
		/// auto driving motion command, automatically route xbots to their target positions
		/// </summary>
		/// <param name="nXbots">number of xbots specified in this command, up to 78</param>
		/// <param name="RoutingType">labelled (0) or unlabelled (1) routing</param>
		/// <param name="ZoneID">0: system routing; >0: zone based routing (zone must be fenced) </param>
		/// <param name="XbotID">pointer to an array containing the xbot IDs of the xbots specified in this command (ID#s from 1-255)</param>
		/// <param name="PosX">pointer to an array containing the target x positions. (m)</param>
		/// <param name="PosY">pointer to an array containing the target y positions. (m)</param>
		/// <returns>
		/// PMC return
		/// </returns>
		PMCRTN AutoDrivingMotion(PMC_USINT nXbots, PMC_USINT RoutingType, PMC_USINT ZoneID, PMC_USINT* XbotID, PMC_REAL* PosX, PMC_REAL* PosY);
		/// <summary>
		/// Command multiple xbots (1-78) to target positions (user set velocity and acceleration). Pathing is determined by PMC
		/// </summary>
		/// <param name="nXbots">number of xbots specified in this command, up to 78</param>
		/// <param name="RoutingType">labelled (0) or unlabelled (1) routing</param>
		/// <param name="ZoneID">0: system routing; >0: zone based routing (zone must be fenced) </param>
		/// <param name="XbotID">pointer to an array containing the xbot IDs of the xbots specified in this command (ID#s from 1-255)</param>
		/// <param name="PosX">pointer to an array containing the target x positions. (m)</param>
		/// <param name="PosY">pointer to an array containing the target y positions. (m)</param>
		/// <param name="MaxVel">Maximum velocity of the xbots during the command (m/s)</param>
		/// <param name="MaxAcc">Maximum acceleration of the xbots during the command (m/s^2)</param>
		/// <returns></returns>
		PMCRTN AutoDrivingMotionVelocity(PMC_USINT nXbots, PMC_USINT RoutingType, PMC_USINT ZoneID, PMC_USINT* XbotID, PMC_REAL* PosX, PMC_REAL* PosY, PMC_REAL MaxVel, PMC_REAL MaxAcc);
		/// <summary>
		/// activates an autoloading zone
		/// </summary>
		/// <param name="ZoneID">zone ID: >0 for creation/activation/deactivation; >=0 with delete zone(s) (ID# from 0-255)</param>
		/// <param name="NextXbotID">Xbot ID to be assigned to next received xbot: 0-PMC assigned, >=1-user assigned</param>
		/// <param name="NextXbotOrientation">Orientation assigned to next received xbot (only user assigned ID)(0-0 deg;1-90 deg;2-180 deg;3-270 deg)</param>
		/// <returns></returns>
		PMCRTN AutoLoadingZone_Activate(PMC_USINT ZoneID, PMC_USINT NextXbotID, PMC_USINT NextXbotOrientation);
		/// <summary>
		/// Creates a new autoloading zone
		/// </summary>
		/// <param name="ZoneID">zone ID: (1-255)</param>
		/// <param name="ZoneMode">0: unloading zone, 1: loading zone</param>
		/// <param name="UnloadingMode">0: soft-landing mode, 1: hard-landing mode</param>
		/// <param name="LoadingDirection">0: auto, 1: loading from +X, 2: loading from -X, 3: loading from +Y, 4: loading from -Y </param>
		/// <param name="LoadFailResetMode">0: auto reset loading zone, 1: manual reset loading zone</param>
		/// <param name="ZoneCenterX">X coordinate of zone center at flyway boundary (m)</param>
		/// <param name="ZoneCenterY">Y coordinate of zone center at flyway boundary (m)</param>
		/// <param name="ZoneLength">zone length in loading/unloading direction (m)</param>
		/// <param name="ZoneWidth">zone width (m)</param>
		/// <param name="MaxXbotSizeX">max mover size in X (m)</param>
		/// <param name="MaxXbotSizeY">max mover size in Y (m)</param>
		/// <param name="MaxVel">max travelling speed inside the zone (m/s)</param>
		/// <param name="MaxAcc">max travelling acceleration inside the zone (m/s^s)</param>
		/// <param name="MaxHeight">max detection height (m) (0 = no limit)</param>
		/// <returns></returns>
		PMCRTN AutoLoadingZone_Create(PMC_USINT ZoneID, ALZONETYPE ZoneMode, ALZONEUNLOADMODE UnloadingMode, PMC_USINT LoadingDirection, PMC_USINT LoadFailResetMode, PMC_REAL ZoneCenterX, PMC_REAL ZoneCenterY, PMC_REAL ZoneLength, PMC_REAL ZoneWidth, PMC_REAL MaxXbotSizeX, PMC_REAL MaxXbotSizeY, PMC_REAL MaxVel, PMC_REAL MaxAcc, PMC_REAL MaxHeight);
		/// <summary>
		/// Deactivate an autoloading zone
		/// </summary>
		/// <param name="ZoneID">zone ID: 1-255</param>
		/// <returns></returns>
		PMCRTN AutoLoadingZone_Deactivate(PMC_USINT ZoneID);
		/// <summary>
		/// deletes one or all autoloading zone(s)
		/// </summary>
		/// <param name="ZoneID">zone ID: 0 - delete all zone, >0 - delete single zone (0-255)</param>
		/// <returns></returns>
		PMCRTN AutoLoadingZone_Delete(PMC_USINT ZoneID);		
		/// <summary>
		/// Gets abbreviated status of all autoloading zones
		/// </summary>
		/// <returns></returns>
		AutoLoadZoneAllStatusReturn AutoLoadingZone_GetAllStatus();
		/// <summary>
		/// Get status of an autoloading zone
		/// </summary>
		/// <param name="ZoneID">ID of zone that will be released (1-255)</param>
		/// <returns></returns>
		AutoLoadZoneStatusReturn AutoLoadingZone_GetZoneStatus(PMC_USINT ZoneID);
		/// <summary>
		/// Release an autoloading zone exit
		/// </summary>
		/// <param name="ZoneID">ID of zone that will be released (1-255)</param>
		/// <param name="NextXbotID">Xbot ID to be assigned to next received xbot: 0-PMC assigned, >=1-user assigned</param>
		/// <param name="NextXbotOrientation">Orientation assigned to next received xbot (only user assigned ID)(0-0 deg;1-90 deg;2-180 deg;3-270 deg)</param>
		/// <returns></returns>
		PMCRTN AutoLoadingZone_LoadingZoneClear(PMC_USINT ZoneID, PMC_USINT NextXbotID,PMC_USINT NextXbotOrientation);
		/// <summary>
		/// Sends an xbot to an auto loading zone
		/// </summary>
		/// <param name="ZoneID">ID of zone that xbot will be sent to (1-255)</param>
		/// <param name="XbotID">ID of xbot to send to zone (1-255)</param>
		/// <returns></returns>
		PMCRTN AutoLoadingZone_UnloadXbot(PMC_USINT ZoneID, PMC_USINT XbotID);
		/// <summary>
		/// Block the motion buffers of the xbots in a group
		/// </summary>
		/// <param name="GroupID">0: block all groups; >0: block specific group</param>
		/// <param name="IsBlock">False: unblock the group(s); True: block the group(s)</param>
		/// <returns></returns>
		PMCRTN BlockXbotGroup(PMC_USINT GroupID, PMC_BOOL IsBlock);
		/// <summary>
		/// Bond xbots in a group
		/// </summary>
		/// <param name="GroupID"> 0: bond all groups; >0: bond specific group</param>
		/// <param name="IsBond">False: unbond group(s); True: bond group(s)</param>
		/// <param name="Mode">0: decouple mode;1: couple mode</param>
		/// <returns></returns>
		PMCRTN BondXbotGroup(PMC_USINT GroupID, PMC_BOOL IsBond, PMC_USINT Mode);
		/// <summary>
		/// Slave the axes of a slave xbot to the axes of one or more master xbots via cams
		/// </summary>
		/// <param name="cmdLB">label to be associated with this command in the PMC (0-65535)</param>
		/// <param name="Level"> 0:Exit from cam mode;1: Enter cam mode</param>
		/// <param name="SlaveXID">ID# of the slave xbot (ID#s 1-255)</param>
		/// <param name="nAxis">number of axes to cam together (up to 7)</param>
		/// <param name="SlaveAID">slave axes 1: X; 2: Y; 3: Z; 4: Rx; 5: Ry; 6: Rz</param>
		/// <param name="camID">ID#s of the cams to use (ID#s 1-255)</param>
		/// <param name="MasterXID">master xbot ID#s (ID#s 1-255)</param>
		/// <param name="MasterAID">master axes 1: X; 2: Y; 3: Z; 4: Rx; 5: Ry; 6: Rz</param>
		/// <returns></returns>
		PMCRTN CamModeCtrl(PMC_UINT cmdLB, PMC_USINT Level, PMC_USINT SlaveXID, PMC_USINT nAxis, PMC_USINT* SlaveAID, PMC_USINT* camID, PMC_USINT* MasterXID, PMC_USINT* MasterAID);
		/// <summary>
		/// Slave the axes of a slave xbot to the axes of one or more master xbots via cams with more advanced options
		/// </summary>
		/// <param name="cmdLB">label to be associated with this command in the PMC (1-65535)</param>
		/// <param name="Level">0:Exit from cam mode;1: Enter cam mode</param>
		/// <param name="SlaveXID">ID# of the slave xbot (1-255)</param>
		/// <param name="nAxis">number of axes to cam together (up to 4)</param>
		/// <param name="SlaveAID">slave axes 1: X; 2: Y; 3: Z; 4: Rx; 5: Ry; 6: Rz</param>
		/// <param name="camID">ID#s of the cams to use (ID#s 1-255)</param>
		/// <param name="MasterXID">master xbot ID#s (ID#s 1-255)</param>
		/// <param name="MasterAID">master axes 1: X; 2: Y; 3: Z; 4: Rx; 5: Ry; 6: Rz</param>
		/// <param name="MasterAOffset">master axis offsets (m)</param>
		/// <param name="SlaveAOffset">slave axis offsets (m)</param>
		/// <param name="MasterAScaling">master axis scaling, positive only</param>
		/// <param name="SlaveAScaling">slave axis scaling, positive only</param>
		/// <param name="CamMode">cam mode; 0: auto start, 1: cyclic, 2: start once </param>
		/// <param name="MasterOffsetMode">0: absolute, 1: relative</param>
		/// <returns></returns>
		PMCRTN CamModeCtrlAdvanced(PMC_UINT cmdLB, PMC_USINT Level, PMC_USINT SlaveXID, PMC_USINT nAxis, PMC_USINT* SlaveAID, PMC_USINT* camID, PMC_USINT* MasterXID, PMC_USINT* MasterAID, PMC_REAL* MasterAOffset, PMC_REAL* SlaveAOffset, PMC_REAL* MasterAScaling, PMC_REAL* SlaveAScaling, PMC_USINT* CamMode, PMC_USINT* MasterOffsetMode);		
		/// <summary>
		/// Slave the axes of a slave xbot to the axes of one or more master xbots via cams
		/// </summary>
		/// <param name="cmdLB">label to be associated with this command in the PMC</param>
		/// <param name="Level">0: Begin IN End OUT, 1: End IN Begin OUT, 2: Begin IN Begin OUT, 3: End IN End OUT</param>
		/// <param name="SlaveXID">ID# of the slave xbot</param>
		/// <param name="nAxis">number of axes to cam together (up to 4)</param>
		/// <param name="SlaveAID">slave axes 1: X; 2: Y; 3: Z; 4: Rx; 5: Ry; 6: Rz</param>
		/// <param name="camID">ID#s of the cams to use</param>
		/// <param name="MasterXID">master xbot ID#s</param>
		/// <param name="MasterAID">master axes 1: X; 2: Y; 3: Z; 4: Rx; 5: Ry; 6: Rz</param>
		/// <param name="MasterAScaling">master axis scaling, positive only </param>
		/// <param name="SlaveAScaling">slave axis scaling, positive only </param>
		/// <returns></returns>
		PMCRTN CamModeCtrlBufferable(PMC_UINT cmdLB, PMC_USINT Level, PMC_USINT SlaveXID, PMC_USINT nAxis, PMC_USINT* SlaveAID, PMC_USINT* camID, PMC_USINT* MasterXID, PMC_USINT* MasterAID, PMC_REAL* MasterAScaling, PMC_REAL* SlaveAScaling);
		/// <summary>
		/// clear commands stored in a macro
		/// </summary>
		/// <param name="MacroID">ID# of the macro to be cleared. Macro IDs are 128 - 191</param>
		/// <returns></returns>
		PMCRTN ClearMacro(PMC_USINT MacroID);		
		/// <summary>
		/// Configure a PMC digital input to start advanced cam motion (rising edge starts cam; falling edge stops cam)
		/// </summary>
		/// <param name="DiID">digital input ID(1-128)</param>
		/// <param name="SlaveXID">ID# of the slave xbot</param>
		/// <param name="nAxis">number of axes to cam together (up to 4)</param>
		/// <param name="SlaveAID">slave axes 1: X; 2: Y; 3: Z; 4: Rx; 5: Ry; 6: Rz</param>
		/// <param name="camID">ID#s of the cams to use</param>
		/// <param name="MasterXID">master xbot ID#s</param>
		/// <param name="MasterAID">master axes 1: X; 2: Y; 3: Z; 4: Rx; 5: Ry; 6: Rz</param>
		/// <param name="MasterAOffset">master axis offsets (m)</param>
		/// <param name="SlaveAOffset">slave axis offsets (m)</param>
		/// <param name="MasterAScaling">master axis scaling, positive only </param>
		/// <param name="SlaveAScaling">slave axis scaling, positive only</param>
		/// <param name="CamMode">cam mode; 0: auto start, 1: cyclic, 2: start once</param>
		/// <param name="MasterOffsetMode">0: absolute, 1: relative</param>
		/// <returns></returns>
		PMCRTN ConfigDI2Cam(PMC_USINT DiID, PMC_USINT SlaveXID, PMC_USINT nAxis, PMC_USINT* SlaveAID, PMC_USINT* camID, PMC_USINT* MasterXID, PMC_USINT* MasterAID, PMC_REAL* MasterAOffset, PMC_REAL* SlaveAOffset, PMC_REAL* MasterAScaling, PMC_REAL* SlaveAScaling, PMC_USINT* CamMode, PMC_USINT* MasterOffsetMode);
		/// <summary>
		/// Configure PMC digital inputs to quickstop the PMC or a sector
		/// </summary>
		/// <param name="nDI">number of digital inputs to configure (up to 22)</param>
		/// <param name="DiID">Array of digital input ID to configure</param>
		/// <param name="SectorID">Array of sector ID#s quick stop (0 = entire PMC)</param>
		/// <returns></returns>
		PMCRTN ConfigDI2QuickStop(PMC_USINT nDI, PMC_USINT* DiID, PMC_USINT* SectorID);
		/// <summary>
		/// Configure PMC digital inputs to reset PMC digital outputs 
		/// </summary>
		/// <param name="nDI">number of digital inputs to configure (up to 22)</param>
		/// <param name="DiID">Array of digital input ID to configure</param>
		/// <param name="RstDoID">Array of digital output ID#s to reset</param>
		/// <returns></returns>
		PMCRTN ConfigDI2Reset(PMC_USINT nDI, PMC_USINT* DiID, PMC_USINT* RstDoID);
		/// <summary>
		/// Configure PMC digital inputs to trigger xbots to run macros
		/// </summary>
		/// <param name="nDI">number of digital inputs to configure (up to 22)</param>
		/// <param name="DiID">Array of digital input ID to configure</param>
		/// <param name="MacroID">Array of macro ID#s to run. Macro IDs are 128 - 191</param>
		/// <param name="XbotID">Array of ID#s to run the macros</param>
		/// <returns></returns>
		PMCRTN ConfigDI2RunMacro(PMC_USINT nDI, PMC_USINT* DiID, PMC_USINT* MacroID, PMC_USINT* XbotID);
		/// <summary>
		/// Configure PMC digital inputs to act as macro triggers
		/// </summary>
		/// <param name="nDI">Number of digital inputs to configure (up to 22)</param>
		/// <param name="DiID">Array of digital input ID to configure</param>
		/// <returns></returns>
		PMCRTN ConfigDI2Trigger(PMC_USINT nDI, PMC_USINT* DiID);
		/// <summary>
		/// Configure digital outputs to set on command labels
		/// </summary>
		/// <param name="nDO">number of digital outputs to configure (up to 5)</param>
		/// <param name="DoID">array of digital output ID#s to configure</param>
		/// <param name="XbotID">array of xbot ID#s to monitor for the command labels</param>
		/// <param name="CmdLb">array of command labels to set on</param>
		/// <param name="Type_">when to set the digital output 0: Cmd Label Start; 1: Cmd Label End; 2: Cmd Label state</param>
		/// <returns></returns>
		PMCRTN ConfigDO2CmdLb(PMC_USINT nDO, PMC_USINT* DoID, PMC_USINT* XbotID, PMC_UINT* CmdLb, PMC_USINT* Type_);
		/// <summary>
		/// Configure digital outputs to set on xbot displacements
		/// </summary>
		/// <param name="nDO">number of digital outputs to configure (up to 5)</param>
		/// <param name="DoID">array of digital output ID#s to configure</param>
		/// <param name="XbotID">array of xbot ID#s to monitor for displacement</param>
		/// <param name="Type_">0: Positive Cross; 1: Negative Cross; 2: State</param>
		/// <param name="Mode">0: X only; 1: Y only ; 2: XFactor*X + YFactor*Y</param>
		/// <param name="Threshold">Thresholds for output trigger in meters</param>
		/// <param name="XFactor">XFactor used for Mode #2</param>
		/// <param name="YFactor">YFactor used for Mode #2</param>
		/// <returns></returns>
		PMCRTN ConfigDO2Disp(PMC_USINT nDO, PMC_USINT* DoID, PMC_USINT* XbotID, PMC_USINT* Type_, PMC_USINT* Mode, PMC_REAL* Threshold, PMC_REAL* XFactor, PMC_REAL* YFactor);
		/// <summary>
		/// Configure digital outputs to set on xbot force
		/// </summary>
		/// <param name="nDO">number of digital outputs to configure (up to 5)</param>
		/// <param name="DoID">array of digital output ID#s to configure</param>
		/// <param name="XbotID">array of xbot ID#s to monitor for force</param>
		/// <param name="Type_">0: Positive Cross; 1: Negative Cross; 2: State</param>
		/// <param name="AxisID">1: X axis; 2: Y axis; 3: Z axis; 4: Rx; 5: Ry; 6: Rz;</param>
		/// <param name="Threshold">Thresholds for output trigger in Newtons/Newton*Meters</param>
		/// <returns></returns>
		PMCRTN ConfigDO2Force(PMC_USINT nDO, PMC_USINT* DoID, PMC_USINT* XbotID, PMC_USINT* Type_, PMC_USINT* AxisID, PMC_REAL* Threshold);
		/// <summary>
		/// Configure digital outputs to set on xbot motion
		/// </summary>
		/// <param name="nDO">Configure digital outputs to set on xbot motion</param>
		/// <param name="DoID">array of digital output ID#s to configure</param>
		/// <param name="XbotID">array of xbot ID#s to monitor for motion</param>
		/// <param name="Type_">0: Motion Start; 1: Motion End; 2: Motion state</param>
		/// <returns></returns>
		PMCRTN ConfigDO2Motion(PMC_USINT nDO, PMC_USINT* DoID, PMC_USINT* XbotID, PMC_USINT* Type_);
		/// <summary>
		/// Configure digital outputs to set on number of xbots in a zone
		/// </summary>
		/// <param name="nDO">number of digital outputs to configure (up to 5)</param>
		/// <param name="DoID">array of digital output ID#s to configure </param>
		/// <param name="ZoneID">array of Zone ID#s to monitor for number of movers in the zone</param>
		/// <param name="Type_">0: mover count rising about threshold; 1: mover count falling below threshold; 2: mover count higher than threshold</param>
		/// <param name="Threshold">Thresholds for output trigger in number of movers  </param>
		/// <returns></returns>
		PMCRTN ConfigDO2Zone(PMC_USINT nDO, PMC_USINT* DoID, PMC_USINT* ZoneID, PMC_USINT* Type_, PMC_USINT* Threshold);
		/// <summary>
		/// Configure feedback streams for xbots position or force
		/// </summary>
		/// <param name="nStm">number of feedback streams (up to 8)</param>
		/// <param name="StmID">feedback stream ID</param>
		/// <param name="XbotID">xbot ID# for the stream</param>
		/// <param name="FBMode">0: position; 1: force; 2: reference position; 3: tracking error</param>
		/// <returns></returns>
		PMCRTN ConfigFBStream(PMC_USINT nStm, PMC_USINT* StmID, PMC_USINT* XbotID, PMC_USINT* FBMode);
		/// <summary>
		/// Configure extended feedback streams for xbots position or force
		/// </summary>
		/// <param name="nXbots">number of feedback streams (up to 150)</param>
		/// <param name="XbotID">xbot ID# for the stream</param>
		/// <param name="FBMode">0: position; 1: force; 2: reference position; 3: tracking error</param>
		/// <returns></returns>
		PMCRTN ConfigFBStreamExtended(PMC_USINT nXbots, PMC_USINT* XbotID, PMC_USINT* FBMode);
		/// <summary>
		/// Resets PMCControllerType internal state, DO NOT ACTIVATE WHEN PMC IS RUNNING
		/// </summary>
		/// <returns></returns>
		PMCRTN ControllerReset();
		/// <summary>
		/// Read the configuration file (as raw binary) from the PMC
		/// </summary>
		/// <returns></returns>
		ConfigRtn GetConfiguration();
		/// <summary>
		/// Sends a configuration file (as raw binary) to the PMC
		/// </summary>
		/// <param name="nBytes">number of bytes in the configuration file (up to 500,000 bytes)</param>
		/// <param name="bytes">the configuration file as a raw binary</param>
		/// <returns></returns>
		PMCRTN SetConfiguration(PMC_DINT nBytes, PMC_BYTE* bytes);
		/// <summary>
		/// Create a xbot group
		/// </summary>
		/// <param name="GroupID">must be greater than 0</param>
		/// <param name="nXbots">number of xbots to be in the group (up to 32)</param>
		/// <param name="XbotID">xbot ID#s to be in the group</param>
		/// <returns></returns>
		PMCRTN CreateXbotGroup(PMC_USINT GroupID, PMC_USINT nXbots, PMC_USINT* XbotID);
		/// <summary>
		/// Deactivate all Xbots
		/// </summary>
		/// <returns>PMC return indicating whether command is successful</returns>
		PMCRTN DeactivateXbots();		
		/// <summary>
		/// Delete trajectory file
		/// </summary>
		/// <param name="TrajID">0: delete all trajectory files; >0: delete specific trajectory file</param>
		/// <returns></returns>
		PMCRTN DeleteTrajectory(PMC_USINT TrajID);
		/// <summary>
		/// delete a xbot group
		/// </summary>
		/// <param name="GroupID">0: delete all groups; >0: delete specific group</param>
		/// <returns></returns>
		PMCRTN DeleteXbotGroup(PMC_USINT GroupID);		
		/// <summary>
		/// Find all xbots whose centers are within a rectangular area
		/// </summary>
		/// <param name="BottomLeftX">X coordinate of bottom left corner of area (m)</param>
		/// <param name="BottomLeftY">Y coordinate of bottom left corner of area (m)</param>
		/// <param name="TopRightX">X coordinate of top right corner of area (m)</param>
		/// <param name="TopRightY">Y coordinate of top right corner of area (m)</param>
		/// <returns></returns>
		FindXbotsInAreaRtn FindXbotsInArea(PMC_REAL BottomLeftX, PMC_REAL BottomLeftY, PMC_REAL TopRightX, PMC_REAL TopRightY);
		/// <summary>
		/// activate force mode control on a xbot
		/// </summary>
		/// <param name="Mode">0: absolte force mode; 1:relative force mode</param>
		/// <param name="Level">0: Exit from force mode;1: Z force mode;2: X force mode;3: Y force mode;4: XY force mode;5: Z torque mode;8: X force + Z torque;9: Y force + Z torque</param>
		/// <param name="XbotID">xbot ID# of the xbot to activate force mode control</param>
		/// <param name="ForceX">force in X in Newtons</param>
		/// <param name="ForceY">force in Y in Newtons</param>
		/// <param name="ForceZ">force in Z in Newtons</param>
		/// <returns></returns>
		PMCRTN ForceModeCtrl(PMC_USINT Mode, PMC_USINT Level, PMC_USINT Buffer, PMC_USINT XbotID, PMC_REAL ForceX, PMC_REAL ForceY, PMC_REAL ForceZ,PMC_REAL TorqueZ);
		/// <summary>
		/// activate force mode control on a xbot with extended options
		/// </summary>
		/// <param name="Mode">0: absolute force mode; 1:relative force mode</param>
		/// <param name="Level">Set which axises have force mode (0-position,1-force): bit0-Fx, bit1-Fy, bit2-Fz, bit3-Tx, bit4-Ty, bit5-Tz </param>
		/// <param name="Buffer">0: enter/exit force mode immediately; 1:add mode change to motion buffer</param>
		/// <param name="XbotID">xbot ID# of the xbot to activate force mode control</param>
		/// <param name="ForceX">force in X in Newtons</param>
		/// <param name="ForceY">force in Y in Newtons</param>
		/// <param name="ForceZ">force in Z in Newtons</param>
		/// <param name="TorqueX">torque around the X axis in Newton Meters</param>
		/// <param name="TorqueY">torque around the Y axis in Newton Meters</param>
		/// <param name="TorqueZ">torque around the Z axis in Newton Meters</param>
		/// <returns></returns>
		PMCRTN ForceModeCtrlEx(PMC_USINT Mode, PMC_USINT Level, PMC_USINT Buffer, PMC_USINT XbotID, PMC_REAL ForceX, PMC_REAL ForceY, PMC_REAL ForceZ, PMC_REAL TorqueX, PMC_REAL TorqueY, PMC_REAL TorqueZ);
		/// <summary>
		/// Delete G-code file
		/// </summary>
		/// <param name="GCodeID">0: delete all G-code files; >0: delete specific G-code file</param>
		/// <returns></returns>
		PMCRTN DeleteGCode(PMC_USINT GCodeID);
		/// <summary>
		/// Get the IDs of all G-code files on the PMC
		/// </summary>
		/// <returns></returns>
		GCodeIDRtn GetGCodeIDs();
		/// <summary>
		/// Reads a G-code file from the PMC
		/// </summary>
		/// <param name="GCodeID">ID of GCode file to read</param>
		/// <returns>PMC return value + G-code file length + pointer to the G-code file as a raw binary stream</returns>
		GCodeRtn ReadGCode(PMC_USINT GCodeID);
		/// <summary>
		/// Run G-code file
		/// </summary>
		/// <param name="cmdLB">label to be associated with this command in the PMC</param>
		/// <param name="GCodeID">G-code ID# (1-64)</param>
		/// <param name="XbotID">xbot ID# (>0)</param>
		/// <returns></returns>
		PMCRTN RunGCode(PMC_UINT cmdLB, PMC_USINT GCodeID, PMC_USINT XbotID);
		/// <summary>
		/// Send a G-code file to the PMC
		/// </summary>
		/// <param name="GCodeID">GCode ID# (1-64)</param>
		/// <param name="nBytes">number of bytes in the G-code file (up to 200,000 bytes)</param>
		/// <param name="bytes">the G-code file as a raw binary</param>
		/// <returns></returns>
		PMCRTN SaveGCode(PMC_USINT GCodeID, PMC_DINT nBytes, PMC_BYTE* bytes);
		/// <summary>
		/// Get the location of all accidents in the system
		/// </summary>
		/// <returns></returns>
		GetAllAccidentLocationsRtn GetAllAccidentLocations();
		/// <summary>
		/// Get the number and IDs of all accident xbots
		/// </summary>
		/// <returns></returns>
		AccidentXbotRtn GetAllAccidentXbots();
		/// <summary>
		/// Get the status of a border
		/// </summary>
		/// <param name="BorderID">Border ID (>0)</param>
		/// <returns></returns>
		BorderStatusRtn GetBorderStatus(PMC_USINT BorderID);
		/// <summary>
		/// Get status of a Xbot's motion buffer
		/// </summary>
		/// <param name="BorderID">Xbot ID# (>0)</param>
		/// <returns></returns>
		MotionBufferStatusRtn GetBufferStatus(PMC_USINT XbotID);
		/// <summary>
		/// Read the PMC's error log
		/// </summary>
		/// <returns></returns>
		LogRtn GetErrorLog();	
		/// <summary>
		/// Get number of flyway connection errors for each flyway
		/// </summary>
		/// <returns></returns>
		GetFlywayConnectErrorCountsRtn GetFlywayConnectErrorCounts();
		/// <summary>
		/// Get flyway error code
		/// </summary>
		/// <param name="FlywayID">Flyway ID# (>0)</param>
		/// <returns></returns>
		FlywayErrorRtn GetFlywayError(PMC_USINT FlywayID);
		/// <summary>
		/// Reads the serial number of a flyway
		/// </summary>
		/// <param name="FlywayID">Flyway logical ID (>0)</param>
		/// <returns></returns>
		SerialNumRtn GetFlywaySerialNum(PMC_USINT FlywayID);
		/// <summary>
		/// Get temperatures and power consumption of a flyway
		/// </summary>
		/// <param name="FlwID">Flyway ID</param>
		/// <returns></returns>
		FlywayTempRtn GetFlywayTemp(PMC_USINT FlwID);
		/// <summary>
		/// Get status of a xbot group 
		/// </summary>
		/// <param name="GroupID">Group ID >0</param>
		/// <returns></returns>
		GroupStatusRtn GetGroupStatus(PMC_USINT GroupID);
		/// <summary>
		/// Get the number of and IDs of any incoming xbots
		/// </summary>
		/// <returns></returns>
		IncomingXbotsRtn GetIncomingXbots();
		/// <summary>
		/// get status of a macro 
		/// </summary>
		/// <param name="MacroID">Macro ID#</param>
		/// <returns></returns>
		MotionMacroReturn GetMacroStatus(PMC_USINT MacroID);		
		/// <summary>
		/// Get payload of a xbot
		/// </summary>
		/// <param name="XbotID">Xbot ID# (>0)</param>
		/// <returns></returns>
		PayloadRtn GetPayload(PMC_USINT XbotID);		
		/// <summary>
		/// Retreive the latest PMC error codes
		/// </summary>
		/// <returns></returns>
		ErrorCodeRtn GetPMCErrorCode();		
		/// <summary>
		/// Reads the serial number of the PMC
		/// </summary>
		/// <returns></returns>
		SerialNumRtn GetPMCSerialNum();		
		/// <summary>
		/// Get status of the Planar Motor Controller  
		/// </summary>
		/// <returns></returns>
		PMCStatusRtn GetPmcStatus();		
		/// <summary>
		/// Gets the PMC software version number
		/// </summary>
		/// <returns></returns>
		VersionRtn GetPMCVersion();		
		/// <summary>
		/// Get the last 100 warning codes from the PMC
		/// </summary>
		/// <returns></returns>
		WarningsReturn GetPMCWarnings();
		/// <summary>
		/// Get the number of PMNet communication errors for each flyway
		/// </summary>
		/// <returns></returns>
		GetPMNetErrorCountsRtn GetPMNetErrorCounts();
		/// <summary>
		/// Get spaced out xbot positions
		/// </summary>
		/// <returns></returns>
		GetSpacedXbotPositionsRtn GetSpacedXbotPositions();
		/// <summary>
		/// Read the PMC's warning log
		/// </summary>
		/// <returns></returns>
		LogRtn GetWarningLog();	
		/// <summary>
		/// Gets the number of xbots in system and their ID#s (up to 99 xbots)(much slower than ReadXbotIDs)
		/// </summary>
		/// <returns></returns>
		GetXbotIDsRtn GetXbotIDs();
		/// <summary>
		/// Get properties of a xbot
		/// </summary>
		/// <param name="XbotID">xbot ID# (>0)</param>
		/// <returns></returns>
		XbotPropertyReturn GetXbotProperty(PMC_USINT XbotID);		
		/// <summary>
		/// Get status of a xbot
		/// </summary>
		/// <param name="XbotID">xbot ID# (>0)</param>
		/// <param name="FeedbackType">0: Feedback Position 1: Feedback Force</param>
		/// <returns></returns>
		XBotStatusRtn GetXbotStatus(PMC_USINT XbotID, FEEDBACKOPTION FeedbackType);			
		/// <summary>
		/// send a single linear motion to a group of XBots (creates, bonds, and deletes group all in one step)
		/// </summary>
		/// <param name="nXbots">number of xbots to group and move (up to 32)</param>
		/// <param name="XbotID">xbot ID#s</param>
		/// <param name="LeaderXbotID">xbot to be sent to target position</param>
		/// <param name="Mode">0: Absolute position. 1: relative positioning</param>
		/// <param name="Type">0: direct straight line; 1: move in X then Y; 2: move in Y then X</param>
		/// <param name="PosX">X position in meters</param>
		/// <param name="PosY">Y position in meters</param>
		/// <param name="EndVel">ending velocity in meters/second</param>
		/// <param name="MaxVel">maximum velocity in meters/second</param>
		/// <param name="MaxAcc">maximum acceleration in meters/second^2</param>
		/// <param name="Radius">turn radius of corner if Type = 1 or 2</param>
		/// <returns></returns>
		MotionRtn GroupXYMotion(PMC_USINT nXbots, PMC_USINT* XbotID, PMC_USINT LeaderXbotID, POSITIONMODE Mode, LINEARPATHTYPE Type, PMC_REAL PosX, PMC_REAL PosY, PMC_REAL EndVel, PMC_REAL MaxVel, PMC_REAL MaxAcc, PMC_REAL Radius);
		/// <summary>
		/// inject sinusoidal force/torque signals to one or multiple axes of xbot 
		/// </summary>
		/// <param name="Mode">0: turn off noise,1: turn on noise</param>
		/// <param name="XbotID">xbot to inject noise (>0)</param>
		/// <param name="nAxis">number of axises to inject noise into</param>
		/// <param name="AxisIDs">axis IDs to inject noise into (1-X; 2-Y; 3-Z; 4-Rx; 5-Ry; 6-Rz)</param>
		/// <param name="Amplitudes">frequencies of sinusoidal signals (Hz,maximum is 655 Hz,precision of 0.01 Hz)</param>
		/// <param name="Phases">phases of sinusoidal signals (degrees,precision of 0.01 degs)</param>
		/// <param name="Frequencies">frequencies of sinusoidal signals (Hz)</param>
		/// <param name="Durations">durations of sinusoidal signals (s)</param>
		/// <returns></returns>
		PMCRTN InjectNoise(PMC_USINT Mode, PMC_USINT XbotID, PMC_USINT nAxis, PMC_USINT* AxisIDs, PMC_REAL* Amplitudes, PMC_REAL* Frequencies,PMC_REAL* Phases, PMC_REAL* Durations);
		/// <summary>
		/// inject sinusoidal force/torque signals with delays to one or multiple axes of xbot 
		/// </summary>
		/// <param name="Mode">0: turn off noise,1: turn on noise</param>
		/// <param name="XbotID">xbot to inject noise (>0)</param>
		/// <param name="nAxis">number of axises to inject noise into</param>
		/// <param name="AxisIDs">axis IDs to inject noise into (1-X; 2-Y; 3-Z; 4-Rx; 5-Ry; 6-Rz)</param>
		/// <param name="Amplitudes">amplitudes of sinusoidal signals (N or Nm)</param>
		/// <param name="Frequencies">frequencies of sinusoidal signals (Hz,maximum is 655 Hz,precision of 0.01 Hz)</param>
		/// <param name="Phases">phases of sinusoidal signals (degrees,precision of 0.01 degs)</param>
		/// <param name="Durations">durations of sinusoidal signals (s)</param>
		/// <param name="Delays">delays of sinusoidal signals (s,maximum is 655 sec,precision of 0.01 sec)</param>
		/// <returns></returns>
		PMCRTN InjectNoiseDelay(PMC_USINT Mode, PMC_USINT XbotID, PMC_USINT nAxis, PMC_USINT* AxisIDs, PMC_REAL* Amplitudes, PMC_REAL* Frequencies, PMC_REAL* Phases, PMC_REAL* Durations, PMC_REAL* Delays);
		/// <summary>
		/// jog a xbot in short axises (Z, Rx, Ry, or Rz)
		/// </summary>
		/// <param name="XbotID">xbot ID# (>0)</param>
		/// <param name="Enable">0: stop jogging xbot; 1: start jogging xbot</param>
		/// <param name="AxisID">Axis to jog; 0-Z, 1-Rx, 2-Ry, 3-Rz</param>
		/// <param name="Velocity">jogging speed in meters/second or radians/second</param>
		/// <returns></returns>
		PMCRTN JogShortAxis(PMC_USINT XbotID, PMC_BOOL Enable, PMC_USINT AxisID, PMC_REAL Velocity);
		/// <summary>
		/// jog a xbot in x/y plane
		/// </summary>
		/// <param name="XbotID">xbot ID# (>0)</param>
		/// <param name="Enable">0: stop jogging xbot; 1: start jogging xbot</param>
		/// <param name="Direction">direction to jog in radians</param>
		/// <param name="Velocity">jogging speed in meters/second</param>
		/// <param name="Acceleration">jogging acceleration in meters/second^2</param>
		/// <returns></returns>
		PMCRTN JogVelocity(PMC_USINT XbotID, PMC_BOOL Enable, PMC_REAL Direction, PMC_REAL Velocity,PMC_REAL Acceleration);		
		/// <summary>
		/// Levitate/Land a xbot
		/// </summary>
		/// <param name="XbotID">0: leviate/land all xbots; &gt;0: leviate/land specific xbot</param>
		/// <param name="Level">0 = landed, 1 = levitate</param>
		/// <param name="Speed">0 = 1.6s, 1= 0.8s, 2 = 0.4s, 3 = 0.2s, 4 = 0.1s, 5 = 0.05s (all times approximate)</param>
		/// <param name="LandedZForce">Z axis force when the xbot is landed (Newtons)</param>
		/// <returns></returns>
		PMCRTN LevitationCtrl(PMC_USINT XbotID, LEVITATEOPTIONS Level, PMC_USINT Speed, PMC_REAL LandedZForce);
		/// <summary>
		/// Levitate/Land a xbot with control over the speed of the landing/levitation
		/// </summary>
		/// <param name="XbotID">0: leviate/land all xbots; >0: leviate/land specific xbot</param>
		/// <param name="Level">0 = landed, 1 = levitate</param>
		/// <param name="Speed">0 = 1.6s, 1= 0.8s, 2 = 0.4s, 3 = 0.2s, 4 = 0.1s, 5 = 0.05s (all times approximate)</param>
		/// <returns></returns>
		PMCRTN LevitationCtrlSpeed(PMC_USINT XbotID, LEVITATEOPTIONS Level, PMC_USINT Speed);		
		/// <summary>
		/// Assigns the linkage defined to xbots, and starts up the CAM and sun-planet relationship if possible
		/// </summary>
		/// <param name="LinkID">ID of the linkage [1, 64]</param>
		/// <param name="XbotID">Xbot IDs of the slaves</param>
		/// <returns></returns>
		LinkedMover_AssignRtn LinkedMover_Assign(PMC_USINT LinkID, PMC_USINT* XbotID);
		/// <summary>
		/// Define masked axis of mover linkage, where multiple movers' motion is combined via a CAM (linked cams only respond to single axis motion) to actuate an mechanical axis
		/// </summary>
		/// <param name="LinkID">ID of the linkage [1, 64]</param>
		/// <param name="MasterAxis">master axis: 1-X,2-Y,3-Z,4-Rx,5-Ry,6-Rz</param>
		/// <param name="MasterAxisOffset">master axis offset to linkage center(m)</param>
		/// <param name="NumSlaveAxes">number of slave axis (1-8)</param>
		/// <param name="SlaveID">which of the 4 slave xbots to be cammed</param>
		/// <param name="SlaveAxis">slave axes: 1-X,2-Y,3-Z,4-Rx,5-Ry,6-Rz</param>
		/// <param name="SlaveCamID">cams to use</param>
		/// <param name="SlaveAxisOffset">slave axis offsets to link center(m)</param>
		/// <returns></returns>
		PMCRTN LinkedMover_CamSlaves(PMC_USINT LinkID, PMC_USINT MasterAxis, PMC_REAL MasterAxisOffset, PMC_USINT NumSlaveAxes, PMC_USINT* SlaveID, PMC_USINT* SlaveAxis, PMC_USINT* SlaveCamID, PMC_REAL* SlaveAxisOffset);
		/// <summary>
		/// Defines a new mover linkage
		/// </summary>
		/// <param name="LinkID">ID of the linkage [1, 64]</param>
		/// <param name="nSlaves">number of slave xbots in the linkage</param>
		/// <param name="SlaveX">Default X positions of slaves relative to linkage center(m)</param>
		/// <param name="SlaveY">Default Y positions of slaves relative to linkage center(m)</param>
		/// <returns></returns>
		PMCRTN LinkedMover_Define(PMC_USINT LinkID, PMC_USINT nSlaves, PMC_REAL* SlaveX, PMC_REAL* SlaveY);
		/// <summary>
		/// Control a xbots motion buffer
		/// </summary>
		/// <param name="Level">0: Block buffer;1: release buffer;2: clear buffer</param>
		/// <param name="XbotID">0: control all xbot buffers; >0: control specific xbot buffer</param>
		/// <returns></returns>
		PMCRTN MotionBufferCtrl(MOTIONBUFFEROPTIONS Level, PMC_USINT XbotID);		
		/// <summary>
		/// Start absolute ID auto scanning for the entire PMC
		/// </summary>
		/// <param name="ZoneID">0-Scan using all available area in system,>0-scan using area defined by zone ID</param>
		/// <returns></returns>
		PMCRTN MoverID_AutoScan(PMC_USINT ZoneID);		
		/// <summary>
		/// Change absolute IDs
		/// </summary>
		/// <param name="nXbots">number of xbots to change absolute IDs (<=200)</param>
		/// <param name="CurrentIDs">current absolute IDs to change</param>
		/// <param name="NewIDs">new absolute IDs to change to</param>
		/// <returns></returns>
		PMCRTN MoverID_ChangeAbsoluteIDs(PMC_USINT nXbots, PMC_USINT* CurrentIDs, PMC_USINT* NewIDs);		
		/// <summary>
		/// get the closest scanning location of a xbot
		/// </summary>
		/// <param name="XbotID">XbotID Xbot ID# (>0)</param>
		/// <param name="Orientation">Orientation 0-East;1-North;2-West;3-South</param>
		/// <returns></returns>
		MoverID_GetScanLocationRtn MoverID_GetScanLocation(PMC_USINT XbotID, PMC_USINT Orientation);
		/// <summary>
		/// manually scan a single xbot for its absolute ID
		/// </summary>
		/// <param name="XbotID">XbotID Xbot ID# (>0)</param>
		/// <param name="Orientation">Orientation 0-East;1-North;2-West;3-South</param>
		/// <returns></returns>
		PMCRTN MoverID_ManualScan(PMC_USINT XbotID, PMC_USINT Orientation);		
		/// <summary>
		/// read a xbot's scan status
		/// </summary>
		/// <param name="XbotID">XbotID xbot ID#</param>
		/// <returns></returns>
		MoverID_ReadScanStatusRtn MoverID_ReadScanStatus(PMC_USINT XbotID);
		/// <summary>
		/// recover XID after flyway error
		/// </summary>
		/// <returns></returns>
		PMCRTN MoverID_RecoverID();		
		/// <summary>
		/// Gets the mover ID of a xbot on a flyway
		/// </summary>
		/// <param name="FlywayID">logical ID (>0) of the flyway to scan</param>
		/// <returns></returns>
		MoverID_ScanMoverIDRtn MoverID_ScanMoverID(PMC_USINT FlywayID);		
		/// <summary>
		/// Use the known mover ID of xbots to set the absolute IDs
		/// </summary>
		/// <param name="nXbots">number of xbots to set (up to 150)</param>
		/// <param name="XbotID">ID#s of xbots to set absolute IDs</param>
		/// <param name="RFID">RFIDs</param>
		/// <param name="Orientation">Orientation 0-East;1-North;2-West;3-South</param>
		/// <returns></returns>
		PMCRTN MoverID_Set(PMC_USINT nXbots, PMC_USINT* XbotID, PMC_ULINT* RFID, PMC_USINT* Orientation);		
		/// <summary>
		/// Get mover label of a xbot
		/// </summary>
		/// <param name="XbotID">xbot ID# (>0)</param>
		/// <returns></returns>
		MoverLabel_GetLabelRtn MoverLabel_GetLabel(PMC_USINT XbotID);		
		/// <summary>
		/// search for a xbot based on mover label
		/// </summary>
		/// <param name="MoverLabel">mover label to return the ID for</param>
		/// <returns></returns>
		MoverLabel_GetXbotIDRtn MoverLabel_GetXbotID(PMC_UINT MoverLabel);		
		/// <summary>
		/// Set a xbot's mover label
		/// </summary>
		/// <param name="XbotID">Xbot ID to set the label for</param>
		/// <param name="MoverLabel">Number to set the label to</param>
		/// <returns></returns>
		PMCRTN MoverLabel_SetLabel(PMC_USINT XbotID, PMC_UINT MoverLabel);
		/// <summary>
		/// Assign a stereotype to a xbot
		/// </summary>
		/// <param name="StereotypeID">ID of the mover stereotype. [1, 255]</param>
		/// <param name="XbotID">ID of the XBot to assign to. 0 = all XBots</param>
		/// <param name="BufferAssign">0: assign stereotype immediately; 1:add stereotype change to motion buffer</param>
		/// <returns></returns>
		PMCRTN MoverStereotype_Assign(PMC_USINT StereotypeID, PMC_USINT XbotID, PMC_USINT BufferAssign);
		/// <summary>
		/// Define a mover stereotype
		/// </summary>
		/// <param name="MoverType">XBot Type (0 = M3-06,2 = M3-08,4 = M3-09X,5 = M3-09Y,6 = M3-10,8 = M3-11X,9 = M3-11Y,12 = M3-12,14 = M3-13)</param>
		/// <param name="StereotypeID">ID of the mover stereotype. [1, 255]</param>
		/// <param name="StereotypeDefinition">The stereotype definition.</param>
		/// <returns></returns>
		PMCRTN MoverStereotype_Define(MOVERTYPE MoverType, PMC_USINT StereotypeID, MoverStereotypeData StereotypeDefinition);		
		/// <summary>
		/// Read a mover stereotype
		/// </summary>
		/// <param name="MoverType">XBot Type (0 = M3-06,2 = M3-08,4 = M3-09X,5 = M3-09Y,6 = M3-10,8 = M3-11X,9 = M3-11Y,12 = M3-12,14 = M3-13)</param>
		/// <param name="StereotypeID">ID of the mover stereotype. [1, 255]</param>
		/// <returns></returns>
		MoverStereotypeDefinitionReturn MoverStereotype_ReadDef(MOVERTYPE MoverType, PMC_USINT StereotypeID);			
		/// <summary>
		/// Sends a configuration file (as raw binary) to a flyway
		/// </summary>
		/// <param name="FlywayID">ID of flyway to overwrite</param>
		/// <param name="nBytes">number of bytes in the configuration file (up to 1,000 bytes)</param>
		/// <param name="Bytes">the configuration file as a raw binary</param>
		/// <returns></returns>
		PMCRTN OverwriteFlywayConfig(PMC_USINT FlywayID, PMC_INT nBytes, PMC_BYTE* Bytes);
		/// <summary>
		/// pause xbot motion
		/// </summary>
		/// <param name="XbotID">0: all Xbots; >0: specific Xbot</param>
		/// <returns></returns>
		PMCRTN PauseXbots(PMC_USINT XbotID);		
		/// <summary>
		/// begin star-planet motion
		/// </summary>
		/// <param name="Level">0:disconnect from star xbot;1: connect to star xbot;2: connect to star w/ coupled RZ</param>
		/// <param name="StarXID">xbot ID# of the star xbot</param>
		/// <param name="nXbots">number of planet xbots (up to 32)</param>
		/// <param name="PlanetXID">planet xbot ID#s</param>
		/// <returns></returns>
		PMCRTN PlanetMotionCtrl(PLANETOPTIONS Level, PMC_USINT StarXID, PMC_USINT nXbots, PMC_USINT* PlanetXID);		
		/// <summary>
		/// Reads the digital signal from PMC.
		/// </summary>
		/// <param name="DiID">digital input (of the PLC fieldbus) number [1-128]</param>
		/// <returns></returns>
		DigitalSignalRtn ReadDigitalSignalFromPMC(PMC_USINT DiID);		
		/// <summary>
		/// Reads the number of accident xbots.
		/// </summary>
		/// <returns></returns>
		ReadNumXbotsRtn ReadNumAccidentXbots();		
		/// <summary>
		/// read the number of xbots coming from neighbor PMCs
		/// </summary>
		/// <returns></returns>
		ReadNumXbotsRtn ReadNumIncomingXbots();		
		/// <summary>
		/// Read Planar Motor Controller state 
		/// </summary>
		/// <returns></returns>
		PMCStateRtn ReadPmcState();		
		/// <summary>
		/// read xbot stream feedback
		/// </summary>
		/// <param name="StreamChannelID">stream channel ID#</param>
		/// <param name="XBotAxisID">Axis ID. X=1, Y=2, Z=3, RX=4, RY=5, RZ=6</param>
		/// <returns></returns>
		FeedbackRtn ReadStreamFeedback(PMC_USINT StreamChannelID, PMC_USINT XBotAxisID);		
		/// <summary>
		/// read extended xbot stream feedback 
		/// </summary>
		/// <param name="XbotID">stream channel ID#</param>
		/// <param name="XBotAxisID">Axis ID. X=1, Y=2, Z=3, RX=4, RY=5, RZ=6</param>
		/// <returns></returns>
		FeedbackRtn ReadStreamFeedbackExtended(PMC_USINT XbotID, PMC_USINT XBotAxisID);			
		/// <summary>
		/// Read the timestamp of the PMC communication frame
		/// </summary>
		/// <returns></returns>
		ReadTimeStampRtn ReadTimeStamp();
		/// <summary>
		/// Reads the hash (MD5) and other parameters of the trajectory 
		/// </summary>
		/// <param name="TrajID">trajectory to read</param>
		/// <returns></returns>
		TrajectoryHashRtn ReadTrajectoryHash(PMC_USINT TrajID);
		/// <summary>
		/// Reads the number of xbots in system and their ID#s (up to 78 xbots)
		/// </summary>
		/// <returns></returns>
		ReadXbotIDsRtn ReadXbotIDs();		
		/// <summary>
		/// read xbot position
		/// </summary>
		/// <param name="XbotID">xbot ID# (1-127)</param>
		/// <returns></returns>
		ReadXbotPosRtn ReadXbotPos(PMC_USINT XbotID);
		/// <summary>
		/// Use feedback or extended feedback to calculate a xbot's speed (feedback or extended feedback must be configured before hand)
		/// </summary>
		/// <param name="ID">stream/xbot ID (feedback position must update two times before the output is valid)</param>
		/// <returns></returns>
		ReadXbotSpeedRtn ReadXbotSpeed(PMC_USINT ID);
		/// <summary>
		/// read a xbot's state (ID <= 200)
		/// </summary>
		/// <param name="XbotID">xbot ID# (1-200)</param>
		/// <returns></returns>
		ReadXbotStateRtn ReadXbotState(PMC_USINT XbotID);			
		/// <summary>
		/// read a xbot's state (any ID, but with increased latency)
		/// </summary>
		/// <param name="XbotID">xbot ID#</param>
		/// <returns></returns>
		ReadXbotStateRtn ReadXbotStateEx(PMC_USINT XbotID);
		/// <summary>
		/// read a xbot's status
		/// </summary>
		/// <param name="XbotID">xbot ID# (1-127)</param>
		/// <returns></returns>
		ReadXbotStatusRtn ReadXbotStatus(PMC_USINT XbotID);
		/// <summary>
		/// reboot Planar Motor System
		/// </summary>
		/// <returns></returns>
		PMCRTN Reboot();		
		/// <summary>
		/// Recovers an accident xbot
		/// </summary>
		/// <param name="XbotID">ID# of the xbot to be recovered (>0)</param>
		/// <param name="RecoveryMode">0-resume motion after recovery, 1-pause motion(ResumeXbots to continue), 2-clear motion buffer</param>
		/// <param name="ShortAxis">0:resume to default short axis position; 1:resume to previous short axis position</param>
		/// <returns></returns>
		PMCRTN RecoverAccidentXbot(PMC_USINT XbotID, PMC_USINT RecoveryMode, PMC_USINT ShortAxis);
		/// <summary>
		/// unpause xbot motion
		/// </summary>
		/// <param name="XbotID">0: all Xbots; >0: specific Xbot</param>
		/// <returns></returns>
		PMCRTN ResumeXbots(PMC_USINT XbotID);			
		/// <summary>
		/// Get all scanned RFIDs and their corresponding xbot IDs
		/// </summary>
		/// <returns></returns>
		GetXbotRFIDsRtn RFID_GetXbotRFIDs();
		/// <summary>
		/// Resets absolute xbot IDs for all xbots (PMC must be deactivated first)
		/// </summary>
		/// <returns></returns>
		PMCRTN RFID_ResetIDs();
		/// <summary>
		/// rotates xbot around the Z axis to a target position
		/// </summary>
		/// <param name="cmdLB">label to be associated with this command in the PMC</param>
		/// <param name="XbotID">ID# of the xbot to move (>0)</param>
		/// <param name="BoundaryMode">0: high performance, 1: compact</param>
		///	<param name="MotionMode">0: absolute motion, 1: relative motion</param>
		/// <param name="WrapMode">angle wrapping mode; 0:Absolute position (+/- 100 rotations) 1:positive rotation in the range of [0,2PI) -1: negative rotation in the range of [0,2PI]</param>
		/// <param name="Angle">target angle of rotation in radians</param>
		/// <param name="MaxVel">maximum velocity in rads/second</param>
		/// <param name="MaxAcc">maximum acceleration in rads/second^2</param>
		/// <returns></returns>
		MotionRtn RotaryMotion(PMC_UINT cmdLB, PMC_USINT XbotID,PMC_SINT BoundaryMode,PMC_SINT MotionMode, PMC_SINT WrapMode, PMC_REAL Angle, PMC_REAL MaxVel, PMC_REAL MaxAcc);
		/// <summary>
		/// spins xbot around the Z axis for a set time
		/// </summary>
		/// <param name="cmdLB">label to be associated with this command in the PMC.</param>
		/// <param name="XbotID">ID# of the xbot to move (>0)</param>
		/// <param name="BoundaryMode ">0: high performance, 1: compact</param>
		/// <param name="Angle">final angle of rotation after spin is complete in radians</param>
		/// <param name="MaxVel">maximum velocity in rads/second</param>
		/// <param name="MaxAcc">maximum acceleration in rads/second^2</param>
		/// <param name="Duration">minimum duration of spinning in seconds (0 to spin forever or until StopXbot command sent)</param>
		/// <returns></returns>
		MotionRtn RotaryMotionSpin(PMC_UINT cmdLB, PMC_USINT XbotID, PMC_SINT BoundaryMode, PMC_REAL Angle, PMC_REAL MaxVel, PMC_REAL MaxAcc, PMC_REAL Duration);
		/// <summary>
		/// command xbot to run a macro
		/// </summary>
		/// <param name="cmdLB">label to be associated with this command in the PMC</param>
		/// <param name="MacroID">macro ID# (128 - 191)</param>
		/// <param name="XbotID">xbot ID# (>0)</param>
		/// <returns></returns>
		PMCRTN RunMacro(PMC_UINT cmdLB, PMC_USINT MacroID, PMC_USINT XbotID);
		/// <summary>
		/// save macro
		/// </summary>
		/// <param name="MacroID">macro ID# (128 - 191)</param>
		/// <returns></returns>
		PMCRTN SaveMacro(PMC_USINT MacroID);				
		/// <summary>
		/// Send a trajectory file to the PMC, PMC must be in the deactivated state
		/// </summary>
		/// <param name="TrajID">Traj ID# (1-64)</param>
		/// <param name="nBytes">number of bytes in the G-code file (up to 500,000 bytes)</param>
		/// <param name="Axises">what axises are in the trajectory file: bit0=X; bit1=Y; bit2=Z; bit3=Rx; bit4=Ry; bit5=Rz; bit6=DO (1 means axis selected; 0 means not selected)</param>
		/// <param name="TimeInterval">time interval between positions (seconds)</param>
		/// <param name="Bytes">the trajectory file as a raw binary</param>
		/// <returns></returns>
		PMCRTN SaveTrajectory(PMC_USINT TrajID, PMC_DINT nBytes, PMC_USINT Axises, PMC_REAL TimeInterval, PMC_BYTE* Bytes);		
		/// <summary>
		/// Gets the global ID of a xbot on a flyway
		/// </summary>
		/// <param name="FlywayID">logical ID (>0) of the flyway to scan</param>
		/// <returns></returns>
		ScanXIDRtn ScanXID(PMC_USINT FlywayID);
		/// <summary>
		/// Activate a sector
		/// </summary>
		/// <param name="SectorID">sector ID: >0</param>
		/// <returns></returns>
		PMCRTN Sector_Activate(PMC_USINT SectorID);
		/// <summary>
		/// Deactivate  a sector
		/// </summary>
		/// <param name="SectorID">sector ID: >0</param>
		/// <returns></returns>
		PMCRTN Sector_Deactivate(PMC_USINT SectorID);
		/// <summary>
		/// Get the status of a sector
		/// </summary>
		/// <param name="SectorID">sector ID: >0</param>
		/// <returns></returns>
		Sector_GetStatusRtn Sector_GetStatus(PMC_USINT SectorID);
		/// <summary>
		/// Recover a sector, removing the fence around it
		/// </summary>
		/// <param name="SectorID">sector ID: >0</param>
		/// <returns></returns>
		PMCRTN Sector_Recover(PMC_USINT SectorID);
		/// <summary>
		/// Sends the digital signal to PMC.
		/// </summary>
		/// <param name="DoID">digital output (of the PLC fieldbus) ID# [1-128]</param>
		/// <param name="Level">desired digital signal level</param>
		/// <returns>successfully set/reset the digital signal</returns>
		PMC_BOOL SendDigitalSignalToPMC(PMC_USINT DoID, PMC_BOOL Level);	
		/// <summary>
		/// Enter and leave PMC service mode
		/// </summary>
		/// <param name="Level">0: enter service mode;1: exit service mode</param>
		/// <returns></returns>
		PMCRTN ServiceModeCtrl(PMC_USINT Level);	
		/// <summary>
		/// Sets the PMC date and time used for logging 
		/// </summary>
		/// <param name="Year">year (AD/CE)</param>
		/// <param name="Month">month (one indexed, January = 1)</param>
		/// <param name="Day">day (one indexed, first day of the month = 1)</param>
		/// <param name="Hour">hour (24 hour clock)</param>
		/// <param name="Minute">minute</param>
		/// <param name="Second">second</param>
		/// <returns></returns>
		PMCRTN SetDateAndTime(PMC_UINT Year, PMC_UINT Month, PMC_UINT Day, PMC_UINT Hour, PMC_UINT Minute, PMC_UINT Second);
		/// <summary>
		/// Sets the jerk (rate of acceleration change) limit for a xbot
		/// </summary>
		/// <param name="XbotID">xbot ID# (>0)</param>
		/// <param name="AxisID">axis ID (1: X, 2: Y, 3: Z, 4: Rx, 5: Ry, 6:Rz)</param>
		/// <param name="JerkLimit">Jerk limit (m/s^3 or rad/s^3)</param>
		/// <returns></returns>
		PMCRTN SetJerkLimit(PMC_USINT XbotID, PMC_USINT AxisID, PMC_REAL JerkLimit);
		/// <summary>
		/// Initialize orientation of xbot before activation in absolute ID mode
		/// </summary>
		/// <param name="XbotID">xbot ID# (0 = all xbots)</param>
		/// <param name="Orientation">0-0deg;1-90deg;2-180deg;3-270deg </param>
		/// <returns></returns>
		PMCRTN SetXbotOrientation(PMC_USINT XbotID, PMC_USINT Orientation);
		/// <summary>
		/// set a xbot property
		/// </summary>
		/// <param name="nXbots">number of xbots to set a property on (up to 14)</param>
		/// <param name="XbotID">array of xbot ID#</param>
		/// <param name="PropertyID">array of property IDs; 0: mover type (read only); 1: user configured payload; 2: user configured center of gravity in Z; 3: payload X dimension; 5: payload Y dimension; 7: acceleration limit (read only)</param>
		/// <param name="Value">array of property values; 1: user configured payload (kilograms); 2: user configured center of gravity in Z (meters); 3: payload X dimension (meters); 5: payload Y dimension (meters)</param>
		/// <returns></returns>
		PMCRTN SetXbotProperty(PMC_USINT nXbots, PMC_USINT* XbotID, MOVERPROPERTY* PropertyID, PMC_REAL* Value);		
		/// <summary>
		/// move xbot in Z, Rx, Ry, and, Rz
		/// </summary>
		/// <param name="cmdLB">label to be associated with this command in the PMC</param>
		/// <param name="XbotID">ID# of the xbot to move (>0)</param>
		/// <param name="Mode">0:Absolute;1:Relative</param>
		/// <param name="Type_">0: Rotation only;1: Planar Rotation</param>
		/// <param name="PosZ">Z position in meters</param>
		/// <param name="PosRx">rotation about X axis in radians</param>
		/// <param name="PosRy">rotation about Y axis in radians</param>
		/// <param name="PosRz">rotation about Z axis in radians</param>
		/// <param name="VelZ">velocity in Z axis in meters/second</param>
		/// <param name="VelRx">velocity of X axis rotation in radians/second</param>
		/// <param name="VelRy">velocity of Y axis rotation in radians/second</param>
		/// <param name="VelRz">velocity of Z axis rotation in radians/second</param>
		/// <param name="Cx">Rotation center in X when type=1</param>
		/// <param name="Cy">Rotation center in Y when type=1</param>
		/// <returns></returns>
		MotionRtn ShortAxisMotion(PMC_UINT cmdLB, PMC_USINT XbotID, POSITIONMODE Mode, SHORTAXESCENTERMODE Type_, PMC_REAL PosZ, PMC_REAL PosRx, PMC_REAL PosRy, PMC_REAL PosRz, PMC_REAL VelZ, PMC_REAL VelRx, PMC_REAL VelRy, PMC_REAL VelRz, PMC_REAL Cx, PMC_REAL Cy);					
		/// <summary>
		/// Move a xbot in only one axis 
		/// </summary>
		/// <param name="cmdLB">label to be associated with this command in the PMC</param>
		/// <param name="XbotID">ID# of the xbot to move (>0).</param>
		/// <param name="Mode">0: Absolute position. 1: relative positioning</param>
		/// <param name="Priority">0: no priority, 1: high priority; all other xbots that try to enter the path will be blocked until motion is done</param>
		/// <param name="AxisID">axis to move in 1-X, 2-Y, 3-Z, 4-Rx, 5-Ry, 6-Rz </param>
		/// <param name="Pos">position in meters or radians</param>
		/// <param name="EndVel">ending velocity in meters/second or radians/second</param>
		/// <param name="MaxVel">maximum velocity in meters/second or radians/second</param>
		/// <param name="MaxAcc">maximum acceleration in meters/second^2 or radians/second^2</param>
		/// <returns></returns>
		MotionRtn SingleAxisMotion(PMC_UINT cmdLB, PMC_USINT XbotID, POSITIONMODE Mode,PMC_USINT Priority, PMC_USINT AxisID, PMC_REAL Pos, PMC_REAL EndVel, PMC_REAL MaxVel, PMC_REAL MaxAcc);
		/// <summary>
		/// Define a new star wheel module
		/// </summary>
		/// <param name="WheelID">Wheel ID (>0)</param>
		/// <param name="Direction">Wheel rotation direction (0: clockwise/negative Rz, 1: counterclockwise/position Rz)</param>
		/// <param name="MasterXID">Master Xbot ID (must be virtual xbot 100-127)</param>
		/// <param name="DiscRadius">Wheel radius in meters</param>
		/// <param name="MaxDiscSpeed">Maximum disc rotation speed in rad/s</param>
		/// <param name="MaxXbotAcc">Maximum xbot acceleration in m/s^2</param>
		/// <param name="DiscCenterX">Disc center X coordinate in meters</param>
		/// <param name="DiscCenterY">Disc center Y coordinate in meters</param>
		/// <param name="StartX">Xbot starting location X coordinate in meters</param>
		/// <param name="StartY">Xbot starting location Y coordinate in meters</param>
		/// <param name="EndX">Xbot ending location X coordinate in meters</param>
		/// <param name="EndY">Xbot ending location Y coordinate in meters</param>
		/// <param name="SyncStart">Sync section starting angle in radians</param>
		/// <param name="SyncEnd">Sync section ending angle in radians</param>
		/// <param name="NumVials">Number of vials on the disc (1-100)</param>
		/// <param name="VialLocations">Location of the vials on the disc in radians (CCW is positive)</param>
		/// <returns></returns>
		PMCRTN Star_CreateStar(PMC_USINT WheelID, PMC_USINT Direction, PMC_USINT MasterXID, PMC_REAL DiscRadius, PMC_REAL MaxDiscSpeed, PMC_REAL MaxXbotAcc, PMC_REAL DiscCenterX, PMC_REAL DiscCenterY, PMC_REAL StartX, PMC_REAL StartY, PMC_REAL EndX, PMC_REAL EndY, PMC_REAL SyncStart, PMC_REAL SyncEnd, PMC_USINT NumVials, PMC_REAL* VialLocations);		
		/// <summary>
		/// Delete a star wheel module
		/// </summary>
		/// <param name="WheelID">wheel ID:  0 - delete all wheels, >0 - delete single wheel</param>
		/// <returns></returns>
		PMCRTN Star_DeleteStar(PMC_USINT WheelID);		
		/// <summary>
		/// Get the status of a star wheel module
		/// </summary>
		/// <param name="WheelID">wheel ID: >0</param>
		/// <returns></returns>
		StarStatusReturn Star_GetStarStatus(PMC_USINT WheelID);		
		/// <summary>
		/// send an xbot to a star wheel module
		/// </summary>
		/// <param name="WheelID">wheel ID: >0</param>
		/// <param name="XbotID">xbot ID: >0</param>
		/// <returns></returns>
		PMCRTN Star_SendXbotToStar(PMC_USINT WheelID, PMC_USINT XbotID);		
		/// <summary>
		/// Tell a star wheel module that the xbot is clear of the exit
		/// </summary>
		/// <param name="WheelID">Wheel ID: >0</param>
		/// <returns></returns>
		PMCRTN Star_StarExitClear(PMC_USINT WheelID);		
		/// <summary>
		/// activate PMC, levitate all xbots, report number of xbots in system and their ID#s
		/// </summary>
		/// <returns></returns>
		StartUpPMSRtn StartUpPlanarMotorSystem();
		/// <summary>
		/// stop motion of xbot and clear its buffer
		/// </summary>
		/// <param name="XbotID">xbot ID#; 0: all xbots; 1: specific xbot</param>
		/// <returns></returns>
		PMCRTN StopXbots(PMC_USINT XbotID);					
		/// <summary>
		/// configure xbot reference streaming
		/// </summary>
		/// <param name="Level">0:Exit from stream mode;1: Enter stream mode</param>
		/// <param name="nXbots">number of xbots to configure streaming for (up to 9)</param>
		/// <param name="XbotID">array of xbot ID#s</param>
		/// <param name="StmID">stream ID#s</param>
		/// <param name="StmAxis">Bit0: X;Bit1:Y;Bit2:Z;Bit3: Rx; Bit4: Ry; Bit5: Rz</param>
		/// <returns></returns>
		StreamModeCtrlRtn StreamModeCtrl(PMC_USINT Level, PMC_USINT nXbots, PMC_USINT* XbotID, PMC_USINT* StmID, PMC_USINT* StmAxis);
		/// <summary>
		/// send linear motion commands to multiple xbots at once
		/// </summary>
		/// <param name="nXbots">number of xbots to send linear move to (up to 4)</param>
		/// <param name="XbotID">xbot ID#s</param>
		/// <param name="PosX">X positions in meters</param>
		/// <param name="PosY">Y positions in meters</param>
		/// <param name="EndVel">Ending velocities in meters/second</param>
		/// <param name="MaxVel">maximum velocities in meters/second</param>
		/// <param name="MaxAcc">maximum accelerations in meters/second^2</param>
		/// <returns></returns>
		MotionRtn SyncMotion(PMC_USINT nXbots, PMC_USINT* XbotID, PMC_REAL* PosX, PMC_REAL* PosY, PMC_REAL* EndVel, PMC_REAL* MaxVel, PMC_REAL* MaxAcc);		
		/// <summary>
		/// reset the xbot weight to zero
		/// </summary>
		/// <param name="XbotID">xbot ID# (>0)</param>
		/// <returns></returns>
		PMCRTN TareXbot(PMC_USINT XbotID);
		/// <summary>
		/// Enable or disable ReadXbotPos for a virtual xbot
		/// </summary>
		/// <param name="XbotID">xbot ID# (>=100)</param>
		/// <param name="Mode">0-enable ReadXbotPos for this virtual xbot,1-disable ReadXbotPos for this virtual xbot</param>
		/// <returns></returns>
		PMCRTN VirtualXbotReadPosCtrl(PMC_USINT XbotID, PMC_USINT Mode);
		/// <summary>
		/// Teleport a virtual xbot to a given position (virtual xbot must be idle first)
		/// </summary>
		/// <param name="XbotID">xbot ID# (>=100)</param>
		/// <param name="PosX">X position (meters)</param>
		/// <param name="PosY">Y position (meters)</param>
		/// <param name="PosZ">Z position (meters)</param>
		/// <param name="PosRx">Rx position (radians)</param>
		/// <param name="PosRy">Ry position (radians)</param>
		/// <param name="PosRz">Rz position (radians)</param>
		/// <returns></returns>
		PMCRTN VirtualXbotTeleport(PMC_USINT XbotID, PMC_REAL PosX, PMC_REAL PosY, PMC_REAL PosZ, PMC_REAL PosRx, PMC_REAL PosRy, PMC_REAL PosRz);
		/// <summary>
		/// xbot waits until another xbot performs a command with a certain label
		/// </summary>
		/// <param name="cmdLB">label to be associated with this command in the PMC</param>
		/// <param name="XbotID"># of the xbot to wait (>0)</param>
		/// <param name="TriggerXID">Trigger Xbot ID</param>
		/// <param name="TriggerType">0: Begin;1: End; 2: Execute</param>
		/// <param name="TriggerCmdLb">command label to wait for</param>
		/// <param name="LabelType">0: Motion Cmd Label;1: Run Macro Cmd Label</param>
		/// <returns></returns>
		PMCRTN WaitUntilCmdLb(PMC_UINT cmdLB, PMC_USINT XbotID, PMC_USINT TriggerXID, TRIGGERCMDLABELTYPE TriggerType, PMC_UINT TriggerCmdLb, TRIGGERCMDTYPE LabelType);		
		/// <summary>
		/// xbot waits until another xbots reaches a given displacement
		/// </summary>
		/// <param name="cmdLB">label to be associated with this command in the PMC</param>
		/// <param name="XbotID">ID# of the xbot to wait (>0)</param>
		/// <param name="TriggerXID">Trigger Xbot ID</param>
		/// <param name="dispMode">0: X only; 1: Y only; 2: posXFactor*X + posYFactor*Y</param>
		/// <param name="dispType">0: Greater Than;1: Less Than; 2: Positive cross;3: Negative cross</param>
		/// <param name="Threshold">threshold to wait until in meters</param>
		/// <param name="posXFactor">XFactor used for dispMode 2</param>
		/// <param name="posYFactor">YFactor used for dispMode 2</param>
		/// <returns></returns>
		PMCRTN WaitUntilDisp(PMC_UINT cmdLB, PMC_USINT XbotID, PMC_USINT TriggerXID, TRIGGERDISPLACEMENTMODE dispMode, TRIGGERDISPLACEMENTTYPE dispType, PMC_REAL Threshold, PMC_REAL posXFactor, PMC_REAL posYFactor);		
		/// <summary>
		/// xbot waits for a PMC digital input 
		/// </summary>
		/// <param name="cmdLB">label to be associated with this command in the PMC</param>
		/// <param name="XbotID">ID# of the xbot to wait (>0)</param>
		/// <param name="TriggerDI">Trigger Digital ID</param>
		/// <param name="TriggerType">0: Rising Edge;1: Falling Edge; 2: High Level;3: Low Level</param>
		/// <returns></returns>
		PMCRTN WaitUntilFBDI(PMC_UINT cmdLB, PMC_USINT XbotID, PMC_USINT TriggerDI, TRIGGEREDGETYPE TriggerType);		
		/// <summary>
		/// xbot waits for a amount of time
		/// </summary>
		/// <param name="cmdLB">label to be associated with this command in the PMC</param>
		/// <param name="XbotID">ID# of the xbot to wait (>0)</param>
		/// <param name="Delay">time to wait for in seconds</param>
		/// <returns></returns>
		PMCRTN WaitUntilTimeDelay(PMC_UINT cmdLB, PMC_USINT XbotID, PMC_REAL Delay);			
		/// <summary>
		/// xbot waits until the number of xbots in a zone reaches a threshold
		/// </summary>
		/// <param name="cmdLB">label to be associated with this command in the PMC</param>
		/// <param name="XbotID">ID# of the xbot to wait (>0)</param>
		/// <param name="TriggerZoneID">Trigger Zone ID</param>
		/// <param name="TriggerType">0: rising edge; 1: falling edge; 2: greater then; 3: less than</param>
		/// <param name="Threshold">number of xbots threshold to wait until</param>
		/// <returns></returns>
		PMCRTN WaitUntilZone(PMC_UINT cmdLB, PMC_USINT XbotID, PMC_USINT TriggerZoneID, PMC_USINT TriggerType, PMC_USINT Threshold);
		/// <summary>
		/// Measure the weight of an xbot, averaged over a period of time
		/// </summary>
		/// <param name="XbotID">xbot ID# (>0)</param>
		/// <param name="Duration">weighting duration (s)</param>
		/// <param name="ReturnType">0-wait until weighing is complete before returning,1-start weighing only,2-read weighing result only</param>
		/// <returns></returns>
		PayloadRtn WeighXbot(PMC_USINT XbotID, PMC_REAL Duration,PMC_USINT ReturnType);		
		/// <summary>
		/// Write a reference position/force to the PMC
		/// </summary>
		/// <param name="StmID">stream ID#s [1-9]</param>
		/// <param name="AxisID">1: X; 2: Y; 3: Z; 4: Rx; 5: Ry; 6: Rz</param>
		/// <param name="Pos">reference value; meters for position X/Y/Z; radians for position Rx/Ry/Rz; Newtons for force X/Y/Z; Newton*Meters for force Rx,Ry,Rz</param>
		/// <returns>successfully wrote reference stream</returns>
		PMC_BOOL WriteStream(PMC_USINT StmID, PMC_USINT AxisID, PMC_REAL Pos);
		/// <summary>
		/// Changes state of the part that the xbot is carrying 
		/// </summary>
		/// <param name="nXbots">The number of xbots (1-150)</param>
		/// <param name="XbotID">pointer to array of xbot IDs</param>
		/// <param name="PartState">pointer to array of part states</param>
		/// <returns></returns>
		PMCRTN XbotPartCtrl(PMC_USINT nXbots, PMC_USINT* XbotID, PMC_USINT* PartState);
		/// <summary>
		/// linear motion command using SI units (meters, radians)
		/// </summary>
		/// <param name="cmdLB">command label (0-65535)</param>
		/// <param name="XbotID">xbot ID (1-255)</param>
		/// <param name="Mode">0 = absolute position mode; 1 = relative position mode</param>
		/// <param name="Type">0 = direct; 1 = X then Y; 2 = Y then X</param>
		/// <param name="PosX">Target Position X, in meters</param>
		/// <param name="PosY">Target Position Y, in meters</param>
		/// <param name="EndVel">Final speed, in m/s</param>
		/// <param name="MaxVel">Max speed, in m/s</param>
		/// <param name="MaxAcc">Max acceleration, in m/s^2</param>
		/// <param name="Radius">turn radius of corner if Type = 1 or 2</param>
		/// <returns>PMC return value + travel time in seconds</returns>
		MotionRtn XYMotion(PMC_UINT cmdLB, PMC_USINT XbotID, POSITIONMODE Mode, LINEARPATHTYPE Type, PMC_REAL PosX, PMC_REAL PosY, PMC_REAL EndVel, PMC_REAL MaxVel, PMC_REAL MaxAcc,PMC_REAL Radius);		
		/// <summary>
		/// Disable/enable Rz collision checks in a zone
		/// </summary>
		/// <param name="ZoneID">zone ID: >0</param>
		/// <param name="DisableRzCheck">0 = enable Rz collision check, 1 = disable Rz collision check</param>
		/// <returns></returns>
		PMCRTN Zone_CollisionControl(PMC_USINT ZoneID, PMC_USINT DisableRzCheck);
		/// <summary>
		/// Define the area controlled by a zone
		/// </summary>
		/// <param name="ZoneID">zone ID: >0</param>
		/// <param name="EnableZLimit">0: no Z limit on xbots that enter, 1: enable Z limit</param>
		/// <param name="Mode">0: regular zone, >0: intersection zone with # of xbots allowed to enter</param>
		/// <param name="BottomLeftX">X coordinate of bottom left corner of zone (m)</param>
		/// <param name="BottomLeftY">Y coordinate of bottom left corner of zone (m)</param>
		/// <param name="TopRightX">X coordinate of right top corner of zone (m)</param>
		/// <param name="TopRightY">Y coordinate of right top corner of zone (m)</param>
		/// <param name="ZLimit">xbots at height above this will be blocked by fence: ignored if z limit isn't enabled</param>
		/// <returns></returns>
		PMCRTN Zone_DefineZone(PMC_USINT ZoneID,PMC_USINT Mode, PMC_REAL BottomLeftX, PMC_REAL BottomLeftY, PMC_REAL TopRightX, PMC_REAL TopRightY,PMC_USINT EnableZLimit,PMC_REAL ZLimit);		
		/// <summary>
		/// Activate or deactivate the fence around a zone
		/// </summary>
		/// <param name="ZoneID">Zone ID (>0)</param>
		/// <param name="Level">Control level: 0 - fence off, 1 - fence on</param>
		/// <returns></returns>
		PMCRTN Zone_Fencing(PMC_USINT ZoneID, FENCEOPERATION Level);		
		/// <summary>
		/// Get the status of a zone
		/// </summary>
		/// <param name="ZoneID">Zone ID (>0)</param>
		/// <returns></returns>
		ZoneStatusReturn Zone_GetStatus(PMC_USINT ZoneID);	
		/// <summary>
		/// Get the status of a zone with extended information
		/// </summary>
		/// <param name="ZoneID">Zone ID (>0)</param>
		/// <returns></returns>
		ZoneStatusExReturn Zone_GetStatusEx(PMC_USINT ZoneID);
		/// <summary>
		/// Change the speed override settings for a zone
		/// </summary>
		/// <param name="ZoneID">Zone ID (>0)</param>
		/// <param name="OverrideType">0-override with ratio,1-override with absolute speed and acceleration limits</param>
		/// <param name="SpeedOverrideFactor">multiplicative factor for xbot speed in the zone (0-2)</param>
		/// <param name="AccOverrideFactor">multiplicative factor for xbot acceleration in the zone (0.001-2)</param>
		/// <returns></returns>
		PMCRTN Zone_OverrideZone(PMC_USINT ZoneID,PMC_USINT OverrideType, PMC_REAL SpeedOverrideFactor, PMC_REAL AccOverrideFactor);		
		/// <summary>
		/// Place xbots into a deactivated zone in emulation mode
		/// </summary>
		/// <param name="ZoneID">zone ID: >0 </param>
		/// <param name="NumXbots">number of xbots to place (up to 9) <param>
		/// <param name="MoverType">XBot type to place (0 = M3-06,2 = M3-08,4 = M3-09X,5 = M3-09Y,6 = M3-10,8 = M3-11X,9 = M3-11Y,12 = M3-12,14 = M3-13)</param>
		/// <param name="PosX">X position to place the xbot</param>
		/// <param name="PosY">Y position to place the xbot</param>
		/// <returns></returns>
		PMCRTN Zone_PlaceXbotsEmulation(PMC_USINT ZoneID, PMC_USINT NumXbots, PMC_USINT* MoverType, PMC_REAL* PosX, PMC_REAL* PosY);		
		/// <summary>
		/// Remove all xbots from a deactivated zone in emulation mode
		/// </summary>
		/// <param name="ZoneID">zone ID: >0 </param>
		/// <returns></returns>
		PMCRTN Zone_RemoveXbotsEmulation(PMC_USINT ZoneID);
		/// <summary>
		/// activate or deactivate a zone
		/// </summary>
		/// <param name="ZoneID">Zone ID (>0)</param>
		/// <param name="Level">Control level: 0 - deactivate zone, 1 - activate zone</param>
		/// <returns></returns>
		PMCRTN Zone_ZoneCtrl(PMC_USINT ZoneID, ZONEOPERATION Level);
	};
}